# MODAVA Domain Models

This document outlines all key domain models with their fields and relationships as enforced by the database schema and TypeScript interfaces.

## Core User Models

### User (Supabase Auth)
**File:** `src/types/index.ts` (extends Supabase auth.users)
- `id: string` - UUID primary key
- `email: string` - User email address
- `user_metadata.role: string` - User role (vendor, admin, staff, superadmin)
- `created_at: string` - Account creation timestamp
- `updated_at: string` - Last update timestamp

**Relationships:**
- One-to-one with Vendor
- One-to-many with KYC submissions
- One-to-many with admin actions

### Vendor
**File:** `src/types/index.ts`, `database-schema.sql`
- `id: string` - UUID primary key
- `user_id: string` - References User.id (unique)
- `business_name: string | null` - Business name (2-100 chars)
- `phone_number: string | null` - Contact phone
- `email: string` - Business email
- `onboarding_status: string` - 'incomplete' | 'complete'
- `store_link: string | null` - Unique store identifier
- `plan_id: string | null` - References Plan.id
- `kyc_status: string` - 'pending' | 'approved' | 'rejected'
- `referral_code: string | null` - Unique referral identifier
- `feature_flags: jsonb` - Feature access configuration
- `trial_end_date: date | null` - Trial period end
- `created_at: string` - Profile creation timestamp
- `updated_at: string` - Last update timestamp

**Relationships:**
- Many-to-one with User
- Many-to-one with Plan
- One-to-many with Customers
- One-to-many with Invoices
- One-to-many with Reminders
- One-to-many with KYC submissions
- One-to-many with Price reports
- One-to-many with Notifications
- One-to-many with Greetings
- One-to-many with Orders
- One-to-many with Referrals

## Business Models

### Customer
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id (cascade delete)
- `name: string` - Customer full name
- `phone: string | null` - Contact phone
- `email: string | null` - Contact email
- `notes: string | null` - Additional notes
- `is_flagged: boolean` - Global flag status (default: false)
- `created_at: string` - Customer creation timestamp

**Relationships:**
- Many-to-one with Vendor
- One-to-many with Measurements
- One-to-many with CustomerFlags
- One-to-many with Reminders
- One-to-many with Invoices

### Measurement
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `customer_id: string` - References Customer.id (cascade delete)
- `name: string` - Measurement type name
- `value: string` - Measurement value
- `unit: string | null` - Unit of measurement

**Relationships:**
- Many-to-one with Customer

### CustomerFlag
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `customer_id: string` - References Customer.id (cascade delete)
- `vendor_id: string` - References Vendor.id (cascade delete)
- `reason: string` - Flag reason
- `additional_notes: string | null` - Additional context
- `proof_url: string | null` - Evidence link
- `status: string` - 'pending' | 'approved' | 'rejected'
- `created_at: string` - Flag creation timestamp

**Relationships:**
- Many-to-one with Customer
- Many-to-one with Vendor

## Financial Models

### Plan
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `name: string` - Plan name
- `price: numeric` - Plan cost
- `trial_period_days: integer` - Trial duration (default: 0)
- `created_at: string` - Plan creation timestamp

**Relationships:**
- One-to-many with Vendors

### Invoice
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id (cascade delete)
- `customer_id: string | null` - References Customer.id
- `total_amount: numeric` - Invoice total
- `status: string` - 'draft' | 'sent' | 'paid' | 'overdue' | 'cancelled'
- `payment_link: string | null` - Payment URL
- `payment_provider: string | null` - Payment processor
- `created_at: string` - Invoice creation timestamp

**Relationships:**
- Many-to-one with Vendor
- Many-to-one with Customer
- One-to-many with InvoiceItems
- One-to-many with InvoiceAttachments
- One-to-many with Payments

### InvoiceItem
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `invoice_id: string` - References Invoice.id (cascade delete)
- `name: string` - Item name
- `quantity: numeric` - Item quantity (default: 1)
- `unit_price: numeric` - Price per unit
- `notes: string | null` - Additional notes

**Relationships:**
- Many-to-one with Invoice

### Payment
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `invoice_id: string` - References Invoice.id (cascade delete)
- `vendor_id: string` - References Vendor.id (cascade delete)
- `amount: numeric` - Payment amount
- `provider: string` - Payment provider
- `provider_transaction_id: string | null` - External transaction ID
- `status: string` - Payment status
- `received_at: string | null` - Payment timestamp

**Relationships:**
- Many-to-one with Invoice
- Many-to-one with Vendor

## Operational Models

### Reminder
**File:** `database-schema.sql`, `src/modules/reminders/types.ts`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id (cascade delete)
- `customer_id: string` - References Customer.id (cascade delete)
- `message: string` - Reminder message
- `type: string | null` - Reminder type
- `generated_by_ai: boolean` - AI-generated flag (default: false)
- `status: string` - 'draft' | 'scheduled' | 'sent' | 'cancelled'
- `created_at: string` - Creation timestamp
- `scheduled_at: string | null` - Scheduled send time

**Relationships:**
- Many-to-one with Vendor
- Many-to-one with Customer

### KYC
**File:** `database-schema.sql`, `src/modules/kyc/types.ts`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id (cascade delete)
- `document_urls: jsonb` - Document file URLs
- `status: string` - 'pending' | 'approved' | 'rejected'
- `reviewed_by: string | null` - Admin reviewer UUID
- `reviewed_at: string | null` - Review timestamp

**Relationships:**
- Many-to-one with Vendor
- Many-to-one with User (reviewer)

## Market & Pricing Models

### Market
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `name: string` - Market name
- `city: string | null` - Market location
- `description: string | null` - Market description

**Relationships:**
- One-to-many with Sellers
- One-to-many with PriceReports

### Item
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `canonical_name: string` - Standard item name
- `description: string | null` - Item description
- `synonyms: string[]` - Alternative names

**Relationships:**
- One-to-many with PriceReports
- One-to-many with SellerInventory
- One-to-many with OrderItems

### PriceReport
**File:** `database-schema.sql`, `src/modules/marketprices/types.ts`
- `id: string` - UUID primary key
- `item_id: string` - References Item.id (cascade delete)
- `market_id: string` - References Market.id (cascade delete)
- `vendor_id: string` - References Vendor.id (cascade delete)
- `price: numeric` - Reported price
- `unit: string | null` - Price unit
- `notes: string | null` - Additional notes
- `created_at: string` - Report timestamp

**Relationships:**
- Many-to-one with Item
- Many-to-one with Market
- Many-to-one with Vendor

## E-commerce Models

### Seller
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `market_id: string` - References Market.id
- `shop_name: string` - Shop name
- `contact_info: string | null` - Contact details
- `delivery_options: string | null` - Delivery methods
- `approved_status: string` - 'pending' | 'approved' | 'rejected'

**Relationships:**
- Many-to-one with Market
- One-to-many with SellerInventory
- One-to-many with Orders

### Order
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `buyer_vendor_id: string` - References Vendor.id (cascade delete)
- `seller_id: string` - References Seller.id (cascade delete)
- `shopping_list_id: string | null` - References ShoppingList.id
- `status: string` - 'pending' | 'confirmed' | 'shipped' | 'delivered' | 'cancelled'
- `total_price: numeric` - Order total

**Relationships:**
- Many-to-one with Vendor (buyer)
- Many-to-one with Seller
- Many-to-one with ShoppingList
- One-to-many with OrderItems

### OrderItem
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `order_id: string` - References Order.id (cascade delete)
- `item_id: string` - References Item.id
- `quantity: numeric` - Item quantity
- `unit_price: numeric` - Price per unit

**Relationships:**
- Many-to-one with Order
- Many-to-one with Item

## Shopping & Inventory Models

### ShoppingList
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id (cascade delete)
- `name: string | null` - List name
- `notes: string | null` - Additional notes
- `created_at: string` - Creation timestamp

**Relationships:**
- Many-to-one with Vendor
- One-to-many with ShoppingListItem
- One-to-many with Orders

### ShoppingListItem
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `shopping_list_id: string` - References ShoppingList.id (cascade delete)
- `item_name: string` - Item name
- `desired_quantity: numeric` - Desired amount
- `unit: string | null` - Quantity unit
- `notes: string | null` - Additional notes

**Relationships:**
- Many-to-one with ShoppingList

### InventoryItem
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id (cascade delete)
- `item_name: string` - Item name
- `quantity: numeric` - Available quantity (default: 0)
- `unit: string | null` - Quantity unit
- `color: string | null` - Item color
- `notes: string | null` - Additional notes
- `last_updated_at: string` - Last update timestamp

**Relationships:**
- Many-to-one with Vendor

## System & Configuration Models

### AdminSettings
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `payout_threshold_amount: numeric` - Minimum payout amount (default: 0)
- `kyc_required_above_threshold: boolean` - KYC requirement flag (default: true)
- `contributor_discount_rules: jsonb` - Contributor discount configuration
- `referral_bonus_rules: jsonb` - Referral bonus configuration
- `commission_settings: jsonb` - Commission structure
- `created_at: string` - Creation timestamp

### Contributor
**File:** `database-schema.sql`
- `vendor_id: string` - References Vendor.id (primary key, cascade delete)
- `is_contributor: boolean` - Contributor status (default: false)
- `contributor_level: string` - 'basic' | 'advanced' | 'expert'
- `status: string` - 'active' | 'inactive' | 'suspended'
- `discount_percentage: numeric` - Discount rate (default: 0)
- `contribution_requirement: integer` - Monthly contribution target (default: 0)
- `current_month_contributions: integer` - Current month count (default: 0)
- `last_review_date: date | null` - Last review date

**Relationships:**
- One-to-one with Vendor

### Referral
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `referrer_vendor_id: string` - References Vendor.id (cascade delete)
- `referred_vendor_id: string` - References Vendor.id (cascade delete)
- `status: string` - 'pending' | 'approved' | 'rejected'
- `bonus_amount: numeric` - Referral bonus (default: 0)
- `date: string` - Referral timestamp

**Relationships:**
- Many-to-one with Vendor (referrer)
- Many-to-one with Vendor (referred)

## Communication Models

### Notification
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id
- `title: string` - Notification title
- `message: string` - Notification content
- `sent_at: string` - Send timestamp
- `is_read: boolean` - Read status (default: false)

**Relationships:**
- Many-to-one with Vendor

### Greeting
**File:** `database-schema.sql`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id
- `message: string` - Greeting message
- `time_of_day: string | null` - Time context
- `created_at: string` - Creation timestamp

**Relationships:**
- Many-to-one with Vendor

## Feature & Access Models

### Feature
**File:** `src/modules/subscriptions/types.ts`
- `id: string` - UUID primary key
- `name: string` - Feature name
- `key: string` - Feature identifier
- `description: string` - Feature description
- `is_active: boolean` - Feature availability
- `created_at: string` - Creation timestamp

**Relationships:**
- Many-to-many with Plans (via plan_features)

### Subscription
**File:** `src/modules/subscriptions/types.ts`
- `id: string` - UUID primary key
- `vendor_id: string` - References Vendor.id
- `plan_id: string` - References Plan.id
- `status: string` - 'active' | 'cancelled' | 'expired' | 'trial'
- `start_date: string` - Subscription start
- `end_date: string | null` - Subscription end
- `trial_end_date: string | null` - Trial period end
- `created_at: string` - Creation timestamp

**Relationships:**
- Many-to-one with Vendor
- Many-to-one with Plan

## UNKNOWN Models

The following models are referenced in the code but their complete structure is unclear:

- **SupportTicket** - Referenced in admin controller but schema not found
- **Feature** - Referenced in subscriptions but complete schema not found
- **Plan** - Referenced in subscriptions but complete schema not found

**File paths to investigate:**
- `src/modules/subscriptions/types.ts`
- `src/modules/admin/types.ts`
- Additional database migration files
